<?php
if (!defined('BASEPATH')) exit('No direct script access allowed');

/*
 * Penda Project
 *
 * @developer	        I-Rock Technologies Limited
 * @copyright	        Copyright (c) 2014 - 2024 Penda
 * @project link		https://penda.irock.co.zm
 * @developer link      https://www.irock.co.zm
 */

/**
 * Class Mdl_Setup
 */
class Mdl_Setup extends CI_Model
{
    public $errors = array();

    /**
     * @return bool
     */
    public function install_tables()
    {
        $file_contents = file_get_contents(APPPATH . 'modules/setup/sql/000_1.0.0.sql');

        $this->execute_contents($file_contents);

        $this->save_version('000_1.0.0.sql');

        if ($this->errors) {
            return false;
        }

        $this->install_default_data();

        $this->install_default_settings();

        return true;
    }

    /**
     * @param string $contents
     */
    private function execute_contents($contents)
    {
        $commands = explode(';', $contents);

        foreach ($commands as $command) {
            if (trim($command)) {
                if (!$this->db->query(trim($command) . ';')) {
                    $this->errors[] = $this->db->_error_message();
                }
            }
        }
    }

    /**
     * @param $sql_file
     */
    private function save_version($sql_file)
    {
        $version_db_array = array(
            'version_date_applied' => time(),
            'version_file' => $sql_file,
            'version_sql_errors' => count($this->errors)
        );

        $this->db->insert('pd_versions', $version_db_array);
    }

    /**
     *
     */
    public function install_default_data()
    {
        $this->db->insert('pd_invoice_groups', array(
                'invoice_group_name' => 'Invoice Default',
                'invoice_group_prefix' => 'INV',
                'invoice_group_next_id' => 1
            )
        );

        $this->db->insert('pd_invoice_groups', array(
                'invoice_group_name' => 'Quote Default',
                'invoice_group_prefix' => 'QUO',
                'invoice_group_next_id' => 1
            )
        );

        $this->db->insert('pd_payment_methods', array(
            'payment_method_name' => 'Cash',
        ));

        $this->db->insert('pd_payment_methods', array(
            'payment_method_name' => 'Credit Card',
        ));
    }

    /**
     *
     */
    private function install_default_settings()
    {
        $this->load->helper('string');

        $default_settings = array(
            'default_language' => $this->session->userdata('pd_lang'),
            'date_format' => 'm/d/Y',
            'currency_symbol' => 'ZMW',
            'currency_symbol_placement' => 'before',
            'currency_code' => 'ZMW',
            'invoices_due_after' => 30,
            'quotes_expire_after' => 15,
            'default_invoice_group' => 3,
            'default_quote_group' => 4,
            'thousands_separator' => ',',
            'decimal_point' => '.',
            'cron_key' => random_string('alnum', 16),
            'tax_rate_decimal_places' => 2,
            'pdf_invoice_template' => 'Penda',
            'pdf_invoice_template_paid' => 'Penda - Paid',
            'pdf_invoice_template_overdue' => 'Penda - overdue',
            'pdf_quote_template' => 'Penda',
            'pdf_expense_template' => 'I-Rock Expense',
            'pdf_quote_template_approved' => 'I-Rock Quotation - Approved',
            'public_invoice_template' => 'Penda_Web',
            'public_quote_template' => 'IPenda_Web',
            'disable_sidebar' => 1,
        );

        foreach ($default_settings as $setting_key => $setting_value) {
            $this->db->where('setting_key', $setting_key);

            if (!$this->db->get('pd_settings')->num_rows()) {
                $db_array = array(
                    'setting_key' => $setting_key,
                    'setting_value' => $setting_value
                );

                $this->db->insert('pd_settings', $db_array);
            }
        }
    }

    /**
     * @return bool
     */
    public function upgrade_tables()
    {
        // Collect the available SQL files
        $sql_files = directory_map(APPPATH . 'modules/setup/sql', true);

        // Sort them so they're in natural order
        sort($sql_files);

        // Unset the installer
        unset($sql_files[0]);

        //create new tables

        $this->db->query('
    CREATE TABLE IF NOT EXISTS pd_bills (
      `bill_id` int NOT NULL AUTO_INCREMENT,
      `bill_name` varchar(255) DEFAULT NULL,
      `bill_description` varchar(255) DEFAULT NULL,
      `bill_date` date DEFAULT NULL,
      `bill_amount` decimal(10,0) DEFAULT NULL,
      `bill_category` int DEFAULT NULL,
      `payment_method` varchar(255) DEFAULT NULL,
      `bill_vendor` varchar(255) DEFAULT NULL,
      `update_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`bill_id`)
    ) ENGINE = MyISAM DEFAULT CHARSET = utf8;
');

$this->db->query('
CREATE TABLE IF NOT EXISTS `pd_userlogs` (
    `id` int(11) NOT NULL AUTO_INCREMENT,
    `session_id` datetime NOT NULL,
    `username` varchar(255) NOT NULL,
    `email` varchar(255) NOT NULL,
    `ip_address` binary(16) DEFAULT NULL,
    `loginTime` datetime DEFAULT NULL,
    `logout` datetime DEFAULT NULL,
    PRIMARY KEY (`id`)
  )  ENGINE = MyISAM DEFAULT CHARSET = utf8;
');

$this->db->query('
CREATE TABLE IF NOT EXISTS  `pd_invoice_sumex` (
    `sumex_id`             INT      NOT NULL AUTO_INCREMENT,
    `sumex_invoice`        INT(11)      NOT NULL,
    `sumex_reason`         INT(11)      NOT NULL,
    `sumex_diagnosis`      VARCHAR(500) NOT NULL,
    `sumex_observations`   VARCHAR(500) NOT NULL,
    `sumex_treatmentstart` DATE         NOT NULL,
    `sumex_treatmentend`   DATE         NOT NULL,
    `sumex_casedate`       DATE         NOT NULL,
    `sumex_casenumber`     VARCHAR(35) DEFAULT NULL,
    PRIMARY KEY (`sumex_id`)
  )
    ENGINE = MyISAM
    DEFAULT CHARSET = utf8;
');

$this->db->query('
    CREATE TABLE IF NOT EXISTS pd_expense_items (
      `item_id` int NOT NULL AUTO_INCREMENT,
      `expense_id` int DEFAULT NULL,
      `item_bill_id` int DEFAULT NULL,
      `item_name` text,
      `item_description` longtext,
      `item_quantity` decimal(10,2) NOT NULL,
      `item_price` decimal(20,2) DEFAULT NULL,
      `item_total` decimal(20,2) DEFAULT NULL,
      `update_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`item_id`)
    ) ENGINE = MyISAM DEFAULT CHARSET = utf8;
');

$this->db->query('
    CREATE TABLE IF NOT EXISTS pd_vendors (
      `vendor_id` int NOT NULL AUTO_INCREMENT,
      `vendor_name` varchar(255) DEFAULT NULL,
      PRIMARY KEY (`vendor_id`)
    ) ENGINE = MyISAM DEFAULT CHARSET = utf8;
');

$this->db->query('
    CREATE TABLE IF NOT EXISTS pd_expenses (
      `expense_id` int NOT NULL AUTO_INCREMENT,
      `expense_name` varchar(255) NOT NULL,
      `user_id` int NOT NULL,
      `expense_status` tinyint NOT NULL DEFAULT 1,
      `expense_date_created` date NOT NULL,
      `expense_date_modified` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (`expense_id`),
      KEY `user_id` (`user_id`) USING BTREE
    ) ENGINE = MyISAM DEFAULT CHARSET = utf8;
');

$this->db->query('
    CREATE TABLE IF NOT EXISTS pd_expenses_recurring (
      `expense_recurring_id` int NOT NULL AUTO_INCREMENT,
      `expense_id` int NOT NULL,
      `recur_start_date` date NOT NULL,
      `recur_end_date` date NOT NULL,
      `recur_frequency` varchar(255) NOT NULL,
      `recur_next_date` date NOT NULL,
      PRIMARY KEY (`expense_recurring_id`),
      KEY `invoice_id` (`expense_id`)
    );
');


        // Loop through the files and take appropriate action
        foreach ($sql_files as $sql_file) {
            if (substr($sql_file, -4) == '.sql') {
                //$this->db->query('DROP TABLE IF EXISTS `pd_versions`');
                $this->db->where('version_file', $sql_file);
                // $update_applied = $this->db->get('pd_versions')->row()->update_applied;
                $update_applied = $this->db->get('pd_versions');

                // if (!$update_applied)
                if (!$update_applied->num_rows()) {
                    $file_contents = file_get_contents(APPPATH . 'modules/setup/sql/' . $sql_file);

                    $this->execute_contents($file_contents);

                    $this->save_version($sql_file);

                    // Check for any required upgrade methods
                    $upgrade_method = 'upgrade_' . str_replace('.', '_', substr($sql_file, 0, -4));

                    if (method_exists($this, $upgrade_method)) {
                        $this->$upgrade_method();
                    }
                }
            }
        }

        if ($this->errors) {
            return false;
        }

        $this->install_default_settings();

        return true;
    }

    /**
     * ===========================================
     * Place upgrade functions here
     * e.g. if table rows have to be converted
     * public function upgrade_010_1_0_1() { ... }
     */

    public function upgrade_006_1_2_0()
    {
        /* Update alert to notify about the changes with invoice deletion and credit invoices
         * but only display the warning when the previous version is 1.1.2 or lower and it's an update
         * therefore check if it's an update, if the time difference between v1.1.2 and v1.2.0 is
         * greater than 100 and if v1.2.0 was not installed within this update process
         */
        $this->db->where_in("version_file", array("006_1.2.0.sql", "005_1.1.2.sql"));
        $versions = $this->db->get('pd_versions')->result();
        $upgrade_diff = $versions[1]->version_date_applied - $versions[0]->version_date_applied;

        if ($this->session->userdata('is_upgrade') && $upgrade_diff > 100 && $versions[1]->version_date_applied > (time() - 100)) {
            $setup_notice = array(
                'type' => 'alert-danger',
                'content' => trans('setup_v120_alert'),
            );
            $this->session->set_userdata('setup_notice', $setup_notice);
        }
    }

    public function upgrade_019_1_4_7()
    {
        /* Update alert to set the session configuration $config['sess_use_database'] = false to true
         * but only display the warning when the previous version is 1.4.6 or lower and it's an update
         * (see above for details)
         */
        $this->db->where_in("version_file", array("018_1.4.6.sql", "019_1.4.7.sql"));
        $versions = $this->db->get('pd_versions')->result();
        $upgrade_diff = $versions[1]->version_date_applied - $versions[0]->version_date_applied;

        if ($this->session->userdata('is_upgrade') && $upgrade_diff > 100 && $versions[1]->version_date_applied > (time() - 100)) {
            $setup_notice = array(
                'type' => 'alert-danger',
                'content' => trans('setup_v147_alert'),
            );
            $this->session->set_userdata('setup_notice', $setup_notice);
        }
    }

    public function upgrade_023_1_5_0()
    {
        $res = $this->db->query('SELECT * FROM pd_custom_fields');
        $drop_columns = array();

        $tables = array(
            'client',
            'invoice',
            'quote',
            'payment',
            'user', 
        );

        if ($res->num_rows()) {
            foreach ($res->result() as $row) {
                $drop_columns[] = array(
                    'field_id' => $row->custom_field_id,
                    'column' => $row->custom_field_column,
                    'table' => $row->custom_field_table,
                );
            }
        }

        // Create tables
        $this->db->query('CREATE TABLE `pd_client_custom_new`
            (
                `client_custom_id` INT NOT NULL PRIMARY KEY AUTO_INCREMENT ,
                `client_id` INT NOT NULL, `client_custom_fieldid` INT NOT NULL,
                `client_custom_fieldvalue` TEXT NULL ,
                UNIQUE (client_id, client_custom_fieldid)
            );'
        );

        $this->db->query('CREATE TABLE `pd_invoice_custom_new`
            (
            `invoice_custom_id` INT NOT NULL PRIMARY KEY AUTO_INCREMENT ,
            `invoice_id` INT NOT NULL, `invoice_custom_fieldid` INT NOT NULL,
            `invoice_custom_fieldvalue` TEXT NULL ,
            UNIQUE (invoice_id, invoice_custom_fieldid)
            );'
        );

        $this->db->query('CREATE TABLE `pd_quote_custom_new`
            (
                `quote_custom_id` INT NOT NULL PRIMARY KEY AUTO_INCREMENT ,
                `quote_id` INT NOT NULL, `quote_custom_fieldid` INT NOT NULL,
                `quote_custom_fieldvalue` TEXT NULL ,
                UNIQUE (quote_id, quote_custom_fieldid)
            );'
        );

        $this->db->query('CREATE TABLE `pd_payment_custom_new`
            (
                `payment_custom_id` INT NOT NULL PRIMARY KEY AUTO_INCREMENT ,
                `payment_id` INT NOT NULL, `payment_custom_fieldid` INT NOT NULL,
                `payment_custom_fieldvalue` TEXT NULL ,
                UNIQUE (payment_id, payment_custom_fieldid)
            );'
        );

        $this->db->query('CREATE TABLE `pd_user_custom_new`
            (
                `user_custom_id` INT NOT NULL PRIMARY KEY AUTO_INCREMENT ,
                `user_id` INT NOT NULL, `user_custom_fieldid` INT NOT NULL,
                `user_custom_fieldvalue` TEXT NULL ,
                UNIQUE (user_id, user_custom_fieldid)
            );'
        );

        // Migrate Data
        foreach ($drop_columns as $value) {
            $res = $this->db->query('SELECT * FROM ' . $value['table']);

            preg_match('/^pd_(.*?)_custom$/i', $value['table'], $matches);
            $table_type = $matches[1];
            $table_name = $value['table'] . '_new';

            if ($res->num_rows()) {
                foreach ($res->result() as $row) { 

                    $escaped_table_type = $this->db->escape($row->{$table_type . '_id'});
                    $escaped_column = $this->db->escape($row->{$value['column']});

                    $query = "INSERT INTO $table_name
                        (" . $table_type . "_id, " . $table_type . "_custom_fieldid, " . $table_type . "_custom_fieldvalue)
                        VALUES (
                            $escaped_table_type,
                            (
                                SELECT custom_field_id
                                FROM pd_custom_fields
                                WHERE pd_custom_fields.custom_field_column = " . $this->db->escape($value['column']) . "
                            ),
                            $escaped_column
                        )";

                    $this->db->query($query);
                }
            }
        }

        // Drop old cloumns, and rename new ones
        foreach ($tables as $table) {
            $this->db->query('DROP TABLE IF EXISTS `pd_' . $table . '_custom`');
            $query = 'RENAME TABLE `pd_' . $table . '_custom_new` TO `pd_' . $table . '_custom`';
            $this->db->query($query);
        }

        $this->db->query('ALTER TABLE pd_custom_fields DROP COLUMN custom_field_column');
    }

    public function upgrade_029_1_5_6()
    {
        // The following code will determine if the pd_users table has an existing user_all_clients column
        // If the table already has the column it will be shown in any user query, so get one now
        $test_user = $this->db->query('SELECT * FROM `pd_users` ORDER BY `user_id` ASC LIMIT 1')->row();

        // Add new user key if applicable
        if (!isset($test_user->user_all_clients)) {
            $this->db->query('ALTER TABLE `pd_users`
              ADD `user_all_clients` INT(1) NOT NULL DEFAULT 0
              AFTER `user_psalt`;'
            );
        }

        // Copy the invoice pdf footer to the new quote pdf footer setting
        $this->load->model('settings/mdl_settings');
        $this->mdl_settings->load_settings();
        $this->load->helper('settings');

        $this->mdl_settings->save('pdf_quote_footer', get_setting('pdf_invoice_footer'));
    }
}
