<?php
if (!defined('BASEPATH')) exit('No direct script access allowed');

/*
 * Penda Project
 *
 * @developer	        I-Rock Technologies Limited
 * @copyright	        Copyright (c) 2014 - 2024 Penda
 * @project link		https://penda.irock.co.zm
 * @developer link      https://www.irock.co.zm
 */
/**
 * Class Mdl_Quotes
 */
class Mdl_Quotes extends Response_Model
{
    public $table = 'pd_quotes';
    public $primary_key = 'pd_quotes.quote_id';
    public $date_modified_field = 'quote_date_modified';

    /**
     * @return array
     */
    public function statuses()
    {
        return array(
            '1' => array(
                'label' => trans('draft'),
                'class' => 'draft',
                'href' => 'quotes/status/draft'
            ),
            '2' => array(
                'label' => trans('sent'),
                'class' => 'sent',
                'href' => 'quotes/status/sent'
            ),
            '3' => array(
                'label' => trans('viewed'),
                'class' => 'viewed',
                'href' => 'quotes/status/viewed'
            ),
            '4' => array(
                'label' => trans('approved'),
                'class' => 'approved',
                'href' => 'quotes/status/approved'
            ),
            '5' => array(
                'label' => trans('rejected'),
                'class' => 'rejected',
                'href' => 'quotes/status/rejected'
            ),
            '6' => array(
                'label' => trans('canceled'),
                'class' => 'canceled',
                'href' => 'quotes/status/canceled'
            )
        );
    }

    public function default_select()
    {
        $this->db->select("
            SQL_CALC_FOUND_ROWS
            pd_users.*,
			pd_clients.*,
			pd_quote_amounts.quote_amount_id,
			IFnull(pd_quote_amounts.quote_item_subtotal, '0.00') AS quote_item_subtotal,
			IFnull(pd_quote_amounts.quote_item_tax_total, '0.00') AS quote_item_tax_total,
			IFnull(pd_quote_amounts.quote_tax_total, '0.00') AS quote_tax_total,
			IFnull(pd_quote_amounts.quote_total, '0.00') AS quote_total,
            pd_invoices.invoice_number,
			pd_quotes.*", false);
    }

    public function default_order_by()
    {
        $this->db->order_by('pd_quotes.quote_id DESC');
    }

    public function default_join()
    {
        $this->db->join('pd_clients', 'pd_clients.client_id = pd_quotes.client_id');
        $this->db->join('pd_users', 'pd_users.user_id = pd_quotes.user_id');
        $this->db->join('pd_quote_amounts', 'pd_quote_amounts.quote_id = pd_quotes.quote_id', 'left');
        $this->db->join('pd_invoices', 'pd_invoices.invoice_id = pd_quotes.invoice_id', 'left');
    }

    /**
     * @return array
     */
    public function validation_rules()
    {
        return array(
            'client_id' => array(
                'field' => 'client_id',
                'label' => trans('client'),
                'rules' => 'required'
            ),
            'quote_date_created' => array(
                'field' => 'quote_date_created',
                'label' => trans('quote_date'),
                'rules' => 'required'
            ),
            'invoice_group_id' => array(
                'field' => 'invoice_group_id',
                'label' => trans('quote_group'),
                'rules' => 'required'
            ),
            'quote_password' => array(
                'field' => 'quote_password',
                'label' => trans('quote_password')
            ),
            'user_id' => array(
                'field' => 'user_id',
                'label' => trans('user'),
                'rule' => 'required'
            )
        );
    }

    /**
     * @return array
     */
    public function validation_rules_save_quote()
    {
        return array(
            'quote_number' => array(
                'field' => 'quote_number',
                'label' => trans('quote') . ' #',
                'rules' => 'is_unique[pd_quotes.quote_number' . (($this->id) ? '.quote_id.' . $this->id : '') . ']'
            ),
            'quote_date_created' => array(
                'field' => 'quote_date_created',
                'label' => trans('date'),
                'rules' => 'required'
            ),
            'quote_date_expires' => array(
                'field' => 'quote_date_expires',
                'label' => trans('due_date'),
                'rules' => 'required'
            ),
            'quote_password' => array(
                'field' => 'quote_password',
                'label' => trans('quote_password')
            )
        );
    }

    /**
     * @param null $db_array
     * @return int|null
     */
    public function create($db_array = null)
    {
        $quote_id = parent::save(null, $db_array);

        // Create an quote amount record
        $db_array = array(
            'quote_id' => $quote_id
        );

        $this->db->insert('pd_quote_amounts', $db_array);

        // Create the default invoice tax record if applicable
        if (get_setting('default_invoice_tax_rate')) {
            $db_array = array(
                'quote_id' => $quote_id,
                'tax_rate_id' => get_setting('default_invoice_tax_rate'),
                'include_item_tax' => get_setting('default_include_item_tax'),
                'quote_tax_rate_amount' => 0
            );

            $this->db->insert('pd_quote_tax_rates', $db_array);
        }

        return $quote_id;
    }

    /**
     * Copies quote items, tax rates, etc from source to target
     * @param int $source_id
     * @param int $target_id
     */
    public function copy_quote($source_id, $target_id)
    {
        $this->load->model('quotes/mdl_quote_items');

        $quote_items = $this->mdl_quote_items->where('quote_id', $source_id)->get()->result();

        foreach ($quote_items as $quote_item) {
            $db_array = array(
                'quote_id' => $target_id,
                'item_tax_rate_id' => $quote_item->item_tax_rate_id,
                'item_name' => $quote_item->item_name,
                'item_description' => $quote_item->item_description,
                'item_quantity' => $quote_item->item_quantity,
                'item_price' => $quote_item->item_price,
                'item_order' => $quote_item->item_order
            );

            $this->mdl_quote_items->save(null, $db_array);
        }

        $quote_tax_rates = $this->mdl_quote_tax_rates->where('quote_id', $source_id)->get()->result();

        foreach ($quote_tax_rates as $quote_tax_rate) {
            $db_array = array(
                'quote_id' => $target_id,
                'tax_rate_id' => $quote_tax_rate->tax_rate_id,
                'include_item_tax' => $quote_tax_rate->include_item_tax,
                'quote_tax_rate_amount' => $quote_tax_rate->quote_tax_rate_amount
            );

            $this->mdl_quote_tax_rates->save(null, $db_array);
        }

        // Copy the custom fields
        $this->load->model('custom_fields/mdl_quote_custom');
        $db_array = $this->mdl_quote_custom->where('quote_id', $source_id)->get()->row_array();

        if (count($db_array) > 2) {
            unset($db_array['quote_custom_id']);
            $db_array['quote_id'] = $target_id;
            $this->mdl_quote_custom->save_custom($target_id, $db_array);
        }
    }

    /**
     * @return array
     */
    public function db_array()
    {
        $db_array = parent::db_array();

        // Get the client id for the submitted quote
        $this->load->model('clients/mdl_clients');
        $cid = $this->mdl_clients->where('pd_clients.client_id', $db_array['client_id'])->get()->row()->client_id;
        $db_array['client_id'] = $cid;

        $db_array['quote_date_created'] = date_to_mysql($db_array['quote_date_created']);
        $db_array['quote_date_expires'] = $this->get_date_due($db_array['quote_date_created']);

        $db_array['notes'] = get_setting('default_quote_notes');

        if (!isset($db_array['quote_status_id'])) {
            $db_array['quote_status_id'] = 1;
        }

        $generate_quote_number = get_setting('generate_quote_number_for_draft');

        if ($db_array['quote_status_id'] === 1 && $generate_quote_number == 1) {
            $db_array['quote_number'] = $this->get_quote_number($db_array['invoice_group_id']);
        } elseif ($db_array['quote_status_id'] != 1) {
            $db_array['quote_number'] = $this->get_quote_number($db_array['invoice_group_id']);
        } else {
            $db_array['quote_number'] = '';
        }

        // Generate the unique url key
        $db_array['quote_url_key'] = $this->get_url_key();

        return $db_array;
    }

    /**
     * @param string $quote_date_created
     */
    public function get_date_due($quote_date_created)
    {
        $quote_date_expires = new DateTime($quote_date_created);
        $quote_date_expires->add(new DateInterval('P' . get_setting('quotes_expire_after') . 'D'));
        return $quote_date_expires->format('Y-m-d');
    }

    /**
     * @param $invoice_group_id
     * @return mixed
     */
    public function get_quote_number($invoice_group_id)
    {
        $this->load->model('invoice_groups/mdl_invoice_groups');
        return $this->mdl_invoice_groups->generate_invoice_number($invoice_group_id);
    }

    /**
     * @return string
     */
    public function get_url_key()
    {
        $this->load->helper('string');
        return random_string('alnum', 32);
    }

    /**
     * @param $invoice_id
     * @return mixed
     */
    public function get_invoice_group_id($invoice_id)
    {
        $invoice = $this->get_by_id($invoice_id);
        return $invoice->invoice_group_id;
    }

    /**
     * @param int $quote_id
     */
    public function delete($quote_id)
    {
        parent::delete($quote_id);

        $this->load->helper('orphan');
        delete_orphans();
    }

    /**
     * @return $this
     */
    public function is_draft()
    {
        $this->filter_where('quote_status_id', 1);
        return $this;
    }

    /**
     * @return $this
     */
    public function is_sent()
    {
        $this->filter_where('quote_status_id', 2);
        return $this;
    }

    /**
     * @return $this
     */
    public function is_viewed()
    {
        $this->filter_where('quote_status_id', 3);
        return $this;
    }

    /**
     * @return $this
     */
    public function is_approved()
    {
        $this->filter_where('quote_status_id', 4);
        return $this;
    }

    /**
     * @return $this
     */
    public function is_rejected()
    {
        $this->filter_where('quote_status_id', 5);
        return $this;
    }

    /**
     * @return $this
     */
    public function is_canceled()
    {
        $this->filter_where('quote_status_id', 6);
        return $this;
    }

    /**
     * Used by guest module; includes only sent and viewed
     *
     * @return $this
     */
    public function is_open()
    {
        $this->filter_where_in('quote_status_id', array(2, 3));
        return $this;
    }

    /**
     * @return $this
     */
    public function guest_visible()
    {
        $this->filter_where_in('quote_status_id', array(2, 3, 4, 5));
        return $this;
    }

    /**
     * @param $client_id
     * @return $this
     */
    public function by_client($client_id)
    {
        $this->filter_where('pd_quotes.client_id', $client_id);
        return $this;
    }

    /**
     * @param $quote_url_key
     */
    public function approve_quote_by_key($quote_url_key)
    {
        $this->db->where_in('quote_status_id', array(2, 3));
        $this->db->where('quote_url_key', $quote_url_key);
        $this->db->set('quote_status_id', 4);
        $this->db->update('pd_quotes');
    }

    /**
     * @param $quote_url_key
     */
    public function reject_quote_by_key($quote_url_key)
    {
        $this->db->where_in('quote_status_id', array(2, 3));
        $this->db->where('quote_url_key', $quote_url_key);
        $this->db->set('quote_status_id', 5);
        $this->db->update('pd_quotes');
    }

    /**
     * @param $quote_id
     */
    public function approve_quote_by_id($quote_id)
    {
        $this->db->where_in('quote_status_id', array(2, 3));
        $this->db->where('quote_id', $quote_id);
        $this->db->set('quote_status_id', 4);
        $this->db->update('pd_quotes');
    }

    /**
     * @param $quote_id
     */
    public function reject_quote_by_id($quote_id)
    {
        $this->db->where_in('quote_status_id', array(2, 3));
        $this->db->where('quote_id', $quote_id);
        $this->db->set('quote_status_id', 5);
        $this->db->update('pd_quotes');
    }

    /**
     * @param $quote_id
     */
    public function mark_viewed($quote_id)
    {
        $this->db->select('quote_status_id');
        $this->db->where('quote_id', $quote_id);

        $quote = $this->db->get('pd_quotes');

        if ($quote->num_rows()) {
            if ($quote->row()->quote_status_id == 2) {
                $this->db->where('quote_id', $quote_id);
                $this->db->set('quote_status_id', 3);
                $this->db->update('pd_quotes');
            }
        }
    }

    public function get_all_quotes(){
        $this->db->select('*');
        $this->db->from('pd_quotes');
        $query = $this->db->get();

        if ($query->num_rows() > 0) {
            return $query->result();
        } else {
            return array(); // Empty array if no results
        }
    }
    public function sent_quotes() {
        $this->db->select('*');
        $this->db->from('pd_quotes');
        $this->db->where('quote_status_id', 2);// Add this line to filter active clients
        $query = $this->db->get();

        if ($query->num_rows() > 0) {
            return $query->result();
        } else {
            return array(); // Empty array if no results
        }
    }
    public function approved_quotes() {
        $this->db->select('*');
        $this->db->from('pd_quotes');
        $this->db->where('quote_status_id', 4); // Add this line to filter approved
        $query = $this->db->get();

        if ($query->num_rows() > 0) {
            return $query->result();
        } else {
            return array(); // Empty array if no results
        }
    }

    public function rejected_quotes() {
        $this->db->select('*');
        $this->db->from('pd_quotes');
        $this->db->where('quote_status_id', 5); // Add this line to filter rejected
        $query = $this->db->get();

        if ($query->num_rows() > 0) {
            return $query->result();
        } else {
            return array(); // Empty array if no results
        }
    }

    public function cancelled_quotes() {
        $this->db->select('*');
        $this->db->from('pd_quotes');
        $this->db->where('quote_status_id', 6); // Add this line to filter cancelled
        $query = $this->db->get();

        if ($query->num_rows() > 0) {
            return $query->result();
        } else {
            return array(); // Empty array if no results
        }
    }


    /**
     * @param $quote_id
     */
    public function mark_sent($quote_id)
    {
        $this->db->select('quote_status_id');
        $this->db->where('quote_id', $quote_id);

        $quote = $this->db->get('pd_quotes');

        if ($quote->num_rows()) {
            if ($quote->row()->quote_status_id == 1) {
                $this->db->where('quote_id', $quote_id);
                $this->db->set('quote_status_id', 2);
                $this->db->update('pd_quotes');
            }
        }
    }

    /**
     * @param $quote_id
     */
    public function generate_quote_number_if_applicable($quote_id)
    {
        $quote = $this->mdl_quotes->get_by_id($quote_id);

        if (!empty($quote)) {
            if ($quote->quote_status_id == 1 && $quote->quote_number == "") {
                // Generate new quote number if applicable
                if (get_setting('generate_quote_number_for_draft') == 0) {
                    $quote_number = $this->mdl_quotes->get_quote_number($quote->invoice_group_id);

                    // Set new quote number and save
                    $this->db->where('quote_id', $quote_id);
                    $this->db->set('quote_number', $quote_number);
                    $this->db->update('pd_quotes');
                }
            }
        }
    }

}
